const { Sequelize, Op } = require('sequelize');
const { views } = require("../../models/dbCon");
const { genResponse, isEmpty, genPagingResponse, getLimitForPagination,generateDatesForMonth } = require("../../commons/commons");
const { HTTP_STATUS_CODES, RESPONSE_MSG } = require("../../commons/constants");

class ViewsServices {
    
    
    getUserCountByVendor = async (req, res) => {
        const vendor_id = req.query.vendor_id;
        const month = req.query.month; // Get the month from query parameters
    
        if (!month) {
            return genResponse(res, HTTP_STATUS_CODES.ERROR, 'Month query parameter is required');
        }
    
        const [year, monthNum] = month.split('-').map(Number);
        if (!year || !monthNum || monthNum < 1 || monthNum > 12) {
            return genResponse(res, HTTP_STATUS_CODES.ERROR, 'Invalid month format. Use YYYY-MM');
        }
    
        try {
            // Define the date range for the specified month
            const startDate = new Date(year, monthNum - 1, 1);
            const endDate = new Date(year, monthNum, 0);
    
            // Query the database to count the total users by vendor_id grouped by day
            const userCounts = await views.findAll({
                attributes: [
                    [Sequelize.fn('DATE', Sequelize.col('createdAt')), 'date'],
                    [Sequelize.fn('COUNT', Sequelize.col('user_id')), 'totalUserCount']
                ],
                where: {
                    vendor_id: vendor_id,
                    createdAt: {
                        [Op.gte]: startDate,
                        [Op.lte]: endDate
                    }
                },
                group: [Sequelize.fn('DATE', Sequelize.col('createdAt'))]
            });
    
            // Query the database to get unique user IDs for the specified date range
            const uniqueUserIds = await views.findAll({
                attributes: [
                    [Sequelize.fn('DISTINCT', Sequelize.col('user_id')), 'user_id']
                ],
                where: {
                    vendor_id: vendor_id,
                    createdAt: {
                        [Op.gte]: startDate,
                        [Op.lte]: endDate
                    }
                }
            });
    
            // Convert query results to a map for easy lookup
            const countsMap = new Map(userCounts.map(record => [record.get('date'), record.get('totalUserCount')]));
    
            // Generate the list of all dates for the specified month
            const allDates = generateDatesForMonth(year, monthNum);
    
            // Create the final result, filling missing dates with counts of 0
            const counts = allDates.map(date => (
                countsMap.get(date) || 0
            ));
    
            // Calculate overall month total user count
            const overallTotalUserCount = counts.reduce((acc, cur) => acc + cur, 0);
    
            // Calculate overall month unique user count
            const overallUniqueUserCount = uniqueUserIds.length;
    
            let result = {
                vendor_id: vendor_id,
                counts: counts,
                overallTotalUserCount: overallTotalUserCount,
                overallUniqueUserCount: overallUniqueUserCount
            };
    
            // Send the result as the response
            return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
        } catch (error) {
            console.error('Error fetching user counts:', error);
            return genResponse(res, HTTP_STATUS_CODES.ERROR, RESPONSE_MSG.INTERNAL_SERVER_ERROR);
        }
    };
    
    addViews = async (req, res) => {
        const { vendorId, viewsData } = req.body;
    
        try {

            if (!vendorId ||!viewsData ||!viewsData.length === 0) {
                return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, '');
            }
            // Prepare the views data with the provided vendor ID
            const preparedViewsData = viewsData.map(view => ({ ...view, vendor_id: vendorId }));
    
            // Bulk insert prepared views into the database
            const createdViews = await views.bulkCreate(preparedViewsData);
    

            if (createdViews.length > 0) {
                return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, createdViews);
              } else {
                return genResponse(res, HTTP_STATUS_CODES.FORBIDDEN, RESPONSE_MSG.SMW, 'Error submitting answers');
              }
        } catch (error) {
            console.error('Error adding views:', error);
            return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, 'INTERNAL SERVER ERROR', results);

        }
    };
         
}
module.exports = ViewsServices;