const router = require("express").Router();
const VendorService = require("../../services/vendorServices/vendorService.js");
const multipart = require("connect-multiparty");
const Verify = require("../../services/VendorVerifyToken.js");

var multipartMiddleware = multipart();
let vendorService = new VendorService();

/**
 * @swagger
 * /api/vendors:
 *   get:
 *     summary: Get Vendors
 *     tags: [Vendors]
 *     parameters:
 *       - in: query
 *         name: catId
 *         description: Category ID of the Vendor
 *         schema:
 *           type: integer
 *       - in: query
 *         name: locationId
 *         description: Location ID of the Vendor
 *         schema:
 *           type: integer
 *       - in: query
 *         name: name
 *         description: Vendor name for filtering
 *         schema:
 *           type: string
 *       - in: query
 *         name: page
 *         description: Page number for pagination | default value 0
 *         schema:
 *           type: integer
 *       - in: query
 *         name: pageSize
 *         description: Number of items per page| default value 10
 *         schema:
 *           type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '404':
 *         description: Subscription plan not found.
 *       '500':
 *         description: Internal server error.
 */

router.get("/", Verify, async function (req, res) {
  try {
    await vendorService.getVendors(req, res);
  } catch (e) {
    console.log(e);
  }
});


/**
 * @swagger
 * /api/vendors/{id}:
 *   get:
 *     summary: Get Vendor By Id for Vendor app
 *     tags: [Vendors]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID or Slug of the vendor
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '404':
 *         description: Subscription plan not found.
 *       '500':
 *         description: Internal server error.
 */
router.get("/:id", Verify, async function (req, res) {
  try {
    await vendorService.getVendorDetailByVendorId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/getQuestions/{CateId}:
 *   get:
 *     summary: Get questions by vendor category ID
 *     tags: [Vendors]
 *     description: Retrieve questions based on the vendor ID and category ID.
 *     parameters:
 *       - in: path
 *         name: CateId
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the category whose questions are to be retrieved.
 *       - in: query
 *         name: vendorId
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the vendor.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: A list of questions.
 *       '404':
 *         description: not found.
 *       '500':
 *         description: Internal server error.
 */

router.get("/getQuestions/:CateId", Verify, async function (req, res) {
  try {
    await vendorService.getQuestionByvendorCateId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/getSubPlan/{vendorId}:
 *   get:
 *     summary: Get SubPlans by vendor  ID
 *     tags: [Vendors]
 *     description: Retrieve SubPlans  based on the vendor ID.
 *     parameters:
 *       - in: path
 *         name: vendorId
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the vendor whose SubPlans are to be retrieved.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: A list of SubPlans.
 *       '404':
 *         description: Subscription plan not found.
 *       '500':
 *         description: Internal server error.
 */
router.get("/getSubPlan/:Id", Verify, async function (req, res) {
  try {
    await vendorService.getsubPlanByvendorId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/getLockedDates/{Id}:
 *   get:
 *     summary: Retrieve locked dates by vendor ID and year
 *     tags: [Vendors]
 *     description: Retrieve locked dates for a specific vendor and optionally for a specific year.
 *     parameters:
 *       - in: path
 *         name: Id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Vendor ID
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: A list of locked dates for the specified vendor and year (if provided).
 *       '404':
 *         description: No locked dates found for the specified vendor and year.
 *       '500':
 *         description: Internal server error.
 */

router.get("/getLockedDates/:Id", Verify, async function (req, res) {
  try {
    await vendorService.getlockedDatesByvendorId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/getAlbums/{vendorId}:
 *   get:
 *     summary: Get Albums by vendor  ID
 *     tags: [Vendors]
 *     description: Retrieve Albums based on the vendor ID.
 *     parameters:
 *       - in: path
 *         name: vendorId
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the vendor whose Albums are to be retrieved.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: A list of Albums.
 *       '404':
 *         description: not found.
 *       '500':
 *         description: Internal server error.
 */
router.get("/getAlbums/:Id", Verify, async function (req, res) {
  try {
    await vendorService.getAlbumByvendorId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/getAlbumsDetail/{albumId}:
 *   get:
 *     summary: Get Albums by album  ID
 *     tags: [Vendors]
 *     description: Retrieve Albums based on the album ID.
 *     parameters:
 *       - in: path
 *         name: albumId
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the album whose Albums are to be retrieved.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: A list of album media.
 *       '404':
 *         description: not found.
 *       '500':
 *         description: Internal server error.
 */
router.get("/getAlbumsDetail/:Id", Verify, async function (req, res) {
  try {
    await vendorService.getAlbumDetailByAlbumId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/submitAnswers:
 *   post:
 *     summary: Submit answers
 *     tags: [Vendors]
 *     description: Endpoint to submit answers provided by vendors
 *     requestBody:
 *        required: true
 *        content:
 *          application/json:
 *            schema:
 *              type: object
 *              properties:
 *                vendorId:
 *                  type: integer
 *                  description: The ID of the vendor submitting the answer.
 *                Answers:
 *                  type: array
 *                  items:
 *                    type: object
 *                    properties:
 *                      answers:
 *                        type: string
 *                        description: The answer provided by the vendor.
 *                      question_id:
 *                        type: integer
 *                        description: The ID of the question being answered.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       200:
 *         description: Answers submitted successfully
 *       400:
 *         description: Bad request, missing required field or empty Answers array
 *       500:
 *         description: Internal server error
 */
router.post("/submitAnswers", Verify, async function (req, res) {
  try {
    await vendorService.submitAnswers(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/createAlbums:
 *   post:
 *     summary: Create new albums
 *     tags: [Vendors]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               vendor_Id:
 *                 type: integer
 *           required:
 *               - name
 *               - vendor_Id
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Album created successfully
 *       '400':
 *         description: Bad request
 *       '500':
 *         description: Internal server error
 */
router.post("/createAlbums", Verify, async function (req, res) {
  try {
    await vendorService.createAlbums(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 *  /api/vendors/updateLockedDatesByVendorId:
 *   post:
 *     summary: Update or insert locked dates by vendor ID
 *     tags: [Vendors]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *               date:
 *                 type: string
 *                 format: date
 *               day:
 *                 type: boolean
 *                 enum: [true, false]
 *               night:
 *                 type: boolean
 *                 enum: [true, false]
 *           required:
 *             - vendorId
 *             - date
 *             - day
 *             - night
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successfully updated or inserted locked dates
 *       '500':
 *         description: Internal server error
 */
router.post("/updateLockedDatesByVendorId", Verify, async function (req, res) {
  try {
    await vendorService.updateLockedDatesByVendorId(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/uploadFiles:
 *   post:
 *     summary: Upload media files
 *     tags: [Vendors]
 *     description: Upload media files for a vendor's album
 *     requestBody:
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *                 required: true
 *                 description: Vendor ID
 *               albumId:
 *                 type: integer
 *                 description: Album ID
 *               subvendorId:
 *                 type: integer
 *                 description: Subvendor ID
 *               imageType:
 *                 type: string
 *                 description: image Type (optional)
 *               files:
 *                 type: array
 *                 items:
 *                   type: string
 *                   format: binary
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Files uploaded successfully
 *       '400':
 *         description: Bad request. Missing required parameters.
 *       '500':
 *         description: Internal server error.
 */
router.post("/uploadFiles", Verify, multipartMiddleware, async (req, res) => {
  await vendorService.uploadMedia(req, res);
});

/**
 * @swagger
 * /api/vendors/vender_media:
 *   delete:
 *     summary: Delete vendor media
 *     tags: [Vendors]
 *     description: Delete multiple vendor media files based on vendor ID and image IDs
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *               imageIds:
 *                 type: array
 *                 items:
 *                   type: integer
 *             required:
 *               - vendorId
 *               - imageIds
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Media files deleted successfully
 *       '400':
 *         description: Bad request. Missing required parameters.
 *       '404':
 *         description: No media files found for the provided vendor ID and image IDs.
 *       '500':
 *         description: Internal server error.
 */
router.delete("/vender_media", Verify, async function (req, res) {
  try {
    await vendorService.deleteVendorMedia(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/DeleteAlbum:
 *   delete:
 *     summary: Delete Album
 *     tags: [Vendors]
 *     description: Delete multiple vendor media files based on vendor ID and image IDs
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *               imageIds:
 *                 type: array
 *                 items:
 *                   type: integer
 *             required:
 *               - vendorId
 *               - imageIds
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: deleted successfully
 *       '400':
 *         description: Bad request. Missing required parameters.
 *       '404':
 *         description: No media files found for the provided vendor ID and image IDs.
 *       '500':
 *         description: Internal server error.
 */
router.delete("/DeleteAlbum", Verify, async function (req, res) {
  try {
    await vendorService.deleteVendorAlbum(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/vendors/UpdateVendor:
 *   put:
 *     summary: Update vendor details
 *     tags: [Vendors]
 *     description: Update details of an existing vendor.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *                 description: The ID of the vendor to update.
 *               businessName:
 *                 type: string
 *                 description: The new business name of the vendor.
 *               name:
 *                 type: string
 *                 description: The new name of the vendor.
 *               businessPhone:
 *                 type: string
 *                 description: The new business phone number of the vendor.
 *               facebookPage:
 *                  type: string
 *                  description: The new facebook link of the vendor.
 *               instagramPage:
 *                  type: string
 *                  description: The new instagram link of the vendor.
 *               websiteLink:
 *                  type: string
 *                  description: The new website link of the vendor.
 *               locationId:
 *                 type: integer
 *                 description: The ID of the new location of the vendor.
 *               address:
 *                 type: string
 *                 description: The new address of the vendor.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: OK
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor not found
 *       '500':
 *         description: Internal server error
 */
router.put("/UpdateVendor", Verify, async function (req, res) {
  try {
    console.log(req.body);
    await vendorService.updateVendorDetail(req, res);
  } catch (e) {
    console.log(e);
  }
});

module.exports = router;
