const { Op } = require("sequelize");
const { users, answers, vendorMedia, vendors, vendorPackages, vendorReviews, albums, locations, addresses, vendorCategories, vendorCatBridge, questions, vendorsAnswers, vendorsLockedDates } = require("../models/dbCon");
const { genResponse, isEmpty, genPagingResponse, getLimitForPagination } = require("../commons/commons");
const { HTTP_STATUS_CODES, RESPONSE_MSG } = require("../commons/constants");
const sequelize = require("../models/dbCon").sequelize;

class UsersService {
  async UpdateUserDetail(req, res) {
    try {
      const { user_Id } = req.body;
      let requestbody = {
        email: username,
        phone: phone,
        city: city,
        user_type: userType,
      };

      if (!user_Id) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      } else {
        const customer = await users.update(requestbody, {
          where: {
            user_id: user_Id,
          },
        });

        if (customer) {
          return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, customer);
        } else {
          return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, "Error Creating User Account", result);
        }
      }
    } catch (e) {
      console.log(e);
      return false;
    }
  }

  async answers(req, res) {
    try {
      const { Answers, user_id } = req.body;

      // Check if answers array is empty
      if (!Answers || Answers.length === 0) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, "No answers provided", "");
      }

      // Check if user_id is an integer
      if (!user_id) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, "User_id is Required", "");
      }

      // Construct an array of objects for bulk insertion
      const bulkInsertData = Answers.map((answers) => ({
        answer: answers.answer,
        question_id: answers.question_id,
        user_id: user_id,
      }));

      // Use bulkCreate to insert all answers in one go
      const results = await answers.bulkCreate(bulkInsertData);

      if (results.length > 0) {
        return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, results);
      } else {
        return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, "INTERNAL SERVER ERROR", results);
      }
    } catch (e) {
      console.log(e);
      return false;
    }
  }

  async getVendorDetails(req, res) {
    try {
      const vendorId = req.params.id;

      if (!vendorId) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "vendorId is required");
      }

      // Fetch vendor details along with media
      const vendorDetails = await vendors.findOne({
        where: { id: vendorId },
        attributes: ["business_name", "business_phone", "f_price", "facebook_page_link", "instagram_page_link", "website_link", "description"],
        include: [
          {
            model: vendorMedia,
            where: {
              is_banner: null,
              image_type: "general",
            },
            attributes: ["sequence", "type", "path", "image_type"],
            limit: 10,
          },
          {
            model: addresses,
            attributes: ["full_address"],
            include: [
              {
                model: locations,
                attributes: ["name"],
              },
            ],
          },
          {
            model: vendorCatBridge,
            as: "vendorCatBridges",
            attributes: ["vendor_category_id", "is_parent"],
            where: { is_parent: true }, // only parent category if needed
            required: false,
          },
        ],
      });

      if (!vendorDetails) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "Vendor not found");
      }

      // Fetch vendor packages
      const packages = await vendorPackages.findAll({
        where: { vendor_id: vendorId },
        attributes: ["name", "details", "price", "booking_price", "createdAt"],
      });

      // Find the minimum price among the packages
      let minPrice = 0;
      if (packages && packages.length > 0) {
        minPrice = packages.reduce((min, pkg) => {
          return pkg.price < min ? pkg.price : min;
        }, packages[0].price);
      }

      // Update the f_price field in the vendorDetails object
      vendorDetails.f_price = minPrice;

      // Fetch albums and their media
      const vendorAlbums = await albums.findAll({
        where: { vendor_id: vendorId },
        attributes: ["name", "id"],
        limit: 4,
        include: [
          {
            model: vendorMedia,
            attributes: ["sequence", "type", "path"],
          },
        ],
      });

      const currentDate = new Date();
      const next12Months = new Date(currentDate);
      next12Months.setMonth(next12Months.getMonth() + 12);
      // Extracting only the year, month, and date parts of the dates
      const currentDateString = new Date(currentDate.getFullYear(), currentDate.getMonth(), currentDate.getDate());
      const next12MonthsString = new Date(next12Months.getFullYear(), next12Months.getMonth(), next12Months.getDate());

      let whereCondition = {
        vendor_id: vendorId,
        date: {
          [Op.gte]: currentDateString, // Date greater than or equal to current date
          [Op.lte]: next12MonthsString, // Date less than or equal to next 12 months
        },
      };

      const lockedDates = await vendorsLockedDates.findAll({
        attributes: ["date", "day", "night", "vendor_id"],
        where: whereCondition,
      });

      let questionsWithAnswers = [];
      if (vendorDetails.vendorCatBridges && vendorDetails.vendorCatBridges.length > 0) {
        const vendorCategoryId = vendorDetails.vendorCatBridges[0].vendor_category_id;

        questionsWithAnswers = await questions.findAll({
          where: { vendor_category_id: vendorCategoryId },
          attributes: [
            "id",
            "question",
            "vendor_category_id",
            "status",
          ],
          include: [
            {
              model: vendorsAnswers,
              attributes: ["id", "answers", "status"],
              where: { vendor_id: vendorId },
              required: false,
            },
          ],
        });
      }

      const result = {
        vendorDetails,
        packages,
        vendorAlbums,
        lockedDates,
        questions: questionsWithAnswers,
      };

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e.message);
    }
  }



  async getVendorReviews(req, res, page = 0, pageSize = 10) {
    try {
      if (!req.params.id) {
        throw new Error("vendorId is required");
      }

      // Calculate offset for pagination
      const offset = (page - 1) * pageSize;

      // Fetch paginated reviews
      const reviews = await vendorReviews.findAll({
        attributes: ["feedback", "rating", "createdAt"],
        where: { vendor_id: req.params.id },
        limit: pageSize,
        offset: offset,
        include: [
          {
            model: users,
            attributes: ["name", "profile_image"],
          },
        ],
      });

      // Fetch all reviews to calculate combined score and count
      const allReviews = await vendorReviews.findAll({
        where: { vendor_id: req.params.id },
      });

      // Calculate combined score and count of all reviews
      const combinedReviews = {
        count: allReviews.length,
        averageScore: allReviews.reduce((acc, review) => acc + review.rating, 0) / (allReviews.length || 1),
      };

      // Calculate total pages
      const totalPages = Math.ceil(allReviews.length / pageSize);

      const result = {
        totalPages,
        currentPage: page,
        reviews,
        combinedReviews,
      };

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e.message);
    }
  }

  async getVendorAlbums(req, res, page = 0, pageSize = 10) {
    try {
      if (!req.params.id) {
        throw new Error("vendorId is required");
      }

      // Calculate offset for pagination
      const offset = (page - 1) * pageSize;

      // Fetch paginated reviews
      const vendorAlbums = await albums.findAll({
        where: { vendor_id: req.params.id },
        attributes: ["name", "id"],
        limit: pageSize,
        offset: offset,
        include: [
          {
            model: vendorMedia,
            attributes: ["sequence", "type", "path"],
          },
        ],
      });

      // Fetch all reviews to calculate combined score and count
      const allAlbums = await albums.findAll({
        where: { vendor_id: req.params.id },
      });

      // Calculate total pages
      const totalPages = Math.ceil(allAlbums.length / pageSize);

      const result = {
        totalPages,
        currentPage: page,
        vendorAlbums,
      };

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e.message);
    }
  }

  async getVendorCategories(req, res) {
    try {
      let categories;

      categories = await vendorCategories.findAll({
        attributes: ["name", "id", "icon"],
        include: [{ model: vendorCategories, as: "children", attributes: ["name", "id"] }],
        where: { parent_id: null },
      });

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, categories);
    } catch (error) {
      console.error("Error fetching vendor categories:", error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, error.message);
    }
  }

  async getHomepageVendors(req, res) {
    try {
      if (!req.params.id) {
        throw new Error("locationId is required");
      }
      const categories = await vendorCategories.findAll({
        where: { parent_id: null },
      });

      const result = [];

      for (const category of categories) {
        // Fetch vendor IDs for each category
        const vendorCategoryBridges = await vendorCatBridge.findAll({
          where: { vendor_category_id: category.id },
        });

        const vendorIds = vendorCategoryBridges.map((bridge) => bridge.vendor_id);

        // Fetch vendors for each category and location
        const categoryVendors = await vendors.findAll({
          attributes: [
            "id",
            "business_name",
            "f_price",
            "phone",
            "business_phone",
            [
              sequelize.literal(`(
            SELECT IFNULL(AVG(rating), 0)
            FROM vendor_reviews AS vr
            WHERE vr.vendor_id = vendors.id
          )`),
              "avgRating",
            ],
          ],
          where: {
            id: vendorIds,
          },
          include: [
            {
              model: addresses,
              attributes: ["full_address"],
              where: { location_id: req.params.id },
            },
            {
              model: vendorMedia,
              attributes: ["path"], // add other required attributes
              where: {
                image_type: "cover",
                id: sequelize.literal(`(
                            SELECT id 
                            FROM vendor_media AS vm 
                            WHERE vm.vendor_id = vendors.id 
                              AND vm.image_type = 'cover'
                            ORDER BY vm.createdAt DESC
                            LIMIT 1
                        )`),
              },
            },
          ],
          limit: 10,
        });

        result.push({
          categoryName: category.name,
          categoryId: category.id,
          vendors: categoryVendors,
        });
      }

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (error) {
      console.error("An Error Occured:", error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, error.message);
    }
  }

  // async getCategoryVendors(req, res) {
  //   try {
  //     const { cat_id, loc_id } = req.query;
  //     const { page = 1, limit = 10 } = req.query; // default to page 1 and limit 10 if not provided

  //     if (!cat_id) {
  //       throw new Error("category_id is required");
  //     }
  //     if (!loc_id) {
  //       throw new Error("location_id is required");
  //     }

  //     const offset = (page - 1) * limit;

  //     // Fetch vendors for the given category and location with pagination
  //     const categoryVendors = await vendors.findAndCountAll({
  //       attributes: [
  //         "id",
  //         "business_name",
  //         "f_price",
  //         "phone",
  //         "business_phone",
  //         [
  //           sequelize.literal(`(
  //             SELECT IFNULL(AVG(rating), 0)
  //             FROM vendor_reviews AS vr
  //             WHERE vr.vendor_id = vendors.id
  //           )`),
  //           "avgRating",
  //         ],
  //       ],
  //       include: [
  //         {
  //           model: vendorCatBridge,
  //           attributes: ["vendor_category_id", "is_parent"],
  //         },
  //         {
  //           model: addresses,
  //           attributes: ["full_address"],
  //           where: { location_id: loc_id },
  //         },
  //         {
  //           model: vendorMedia,
  //           attributes: ["path"], // add other required attributes
  //           where: {
  //             image_type:"cover",
  //             id: sequelize.literal(`(
  //                           SELECT id 
  //                           FROM vendor_media AS vm 
  //                           WHERE vm.vendor_id = vendors.id 
  //                           AND vm.image_type = 'cover'
  //                           ORDER BY vm.createdAt DESC
  //                           LIMIT 1
  //                       )`),
  //           },
  //         },
  //       ],
  //       limit: parseInt(limit),
  //       offset: parseInt(offset),
  //     });

  //     const result = {
  //       categoryId: cat_id,
  //       vendors: categoryVendors.rows
  //         .map((vendor) => {
  //           const { vendorCatBridges, ...rest } = vendor.dataValues;
  //           return {
  //             ...rest,
  //             subcategories: vendorCatBridges
  //               .filter((item) => item.is_parent == 0)
  //               .map((bridge) => ({
  //                 id: bridge.vendor_category_id,
  //               })),
  //           };
  //         })
  //         .filter((vendor) => vendor.subcategories.some((sub) => sub.id == cat_id)),

  //       totalVendors: categoryVendors.count,
  //       totalPages: Math.ceil(categoryVendors.count / limit),
  //       currentPage: page,
  //     };

  //     return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
  //   } catch (error) {
  //     console.error("An Error Occured:", error);
  //     return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, error.message);
  //   }
  // }

  async getCategoryVendors(req, res) {
  try {
    const { cat_id, loc_id, page = 1, limit = 10 } = req.query;

    if (!cat_id) throw new Error("category_id is required");
    if (!loc_id) throw new Error("location_id is required");

    const offset = (page - 1) * limit;

    const categoryVendors = await vendors.findAndCountAll({
      attributes: [
        "id",
        "business_name",
        "f_price",
        "phone",
        "business_phone",
        [
          sequelize.literal(`(
            SELECT IFNULL(AVG(rating), 0)
            FROM vendor_reviews AS vr
            WHERE vr.vendor_id = vendors.id
          )`),
          "avgRating",
        ],
      ],
      include: [
        {
          model: vendorCatBridge,
          attributes: ["vendor_category_id", "is_parent"],
          where: { vendor_category_id: cat_id }, // ✅ filters by category
        },
        {
          model: addresses,
          attributes: ["full_address"],
          where: { location_id: loc_id },
        },
        {
          model: vendorMedia,
          attributes: ["path"],
          where: {
            image_type: "cover",
            id: sequelize.literal(`(
              SELECT id 
              FROM vendor_media AS vm 
              WHERE vm.vendor_id = vendors.id 
              AND vm.image_type = 'cover'
              ORDER BY vm.createdAt DESC
              LIMIT 1
            )`),
          },
        },
      ],
      limit: parseInt(limit),
      offset: parseInt(offset),
    });

    const result = {
      categoryId: cat_id,
      vendors: categoryVendors.rows,
      totalVendors: categoryVendors.count,
      totalPages: Math.ceil(categoryVendors.count / limit),
      currentPage: page,
    };

    return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
  } catch (error) {
    console.error("An Error Occured:", error);
    return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, error.message);
  }
}

  async searchVendors(req, res) {
    try {
      if (!req.query.keyword) {
        throw new Error("Keyword is required");
      }

      const whereConditions = {
        business_name: {
          [Op.like]: `%${req.query.keyword}%`,
        },
      };

      const vendorCatBridgeWhere = { is_parent: 0 };

      if (req.query.catId) {
        vendorCatBridgeWhere.vendor_category_id = req.query.catId;
      }

      const results = await vendors.findAll({
        where: whereConditions,
        attributes: ["id", "business_name"],
        include: [
          {
            model: vendorMedia,
            attributes: ["path"], // add other required attributes
            where: {
              image_type: "cover",
              id: sequelize.literal(`(
                          SELECT id 
                          FROM vendor_media AS vm 
                          WHERE vm.vendor_id = vendors.id 
                            AND vm.image_type = 'cover'
                          ORDER BY vm.createdAt DESC
                          LIMIT 1
                      )`),
            },
          },
          {
            model: vendorCatBridge,
            where: vendorCatBridgeWhere,
            include: [
              {
                model: vendorCategories,
                attributes: ["name"],
              },
            ],
          },
          {
            model: addresses,
            attributes: ["full_address"],
            where: { location_id: req.query.locationId },
          },
        ],
      });

      // Convert Sequelize instances to plain objects
      const plainResults = results.map((result) => result.toJSON());

      const data = plainResults.map((item) => {
        return {
          ...item,
          vendorCategory: item.vendorCatBridges[0]?.vendorCategory?.name,
        };
      });

      data.forEach((item) => delete item.vendorCatBridges);

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, data);
    } catch (error) {
      console.log(error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, error.message);
    }
  }

  async addReview(req, res) {
    try {
      const { userId, vendorId, feedback, rating } = req.body;

      // Check if any required field is missing
      if (!userId || !vendorId || !feedback || !rating) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }
      // Validate the format or content of name, vendor_id if needed
      // Create a new entry in the albums table
      else {
        const result = await vendorReviews.create({
          user_id: userId,
          vendor_id: vendorId,
          feedback: feedback,
          rating: rating,
        });

        return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
      }
    } catch (error) {
      // Handle any internal server errors
      console.error("Error submitting review:", error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, "Error submitting review");
    }
  }
}

module.exports = UsersService;
