const router = require("express").Router();
const VendorPackagesService = require("../../services/vendorServices/VendorPackagesService.js");
const multipart = require('connect-multiparty');
const Verify = require('../../services/VendorVerifyToken.js');



let vendorPackagesService = new VendorPackagesService()

/**
 * @swagger
 * tags:
 *   name: VendorPackages
 *   description: API endpoints for managing Subscription
 */

/**
 * @swagger
 * /api/packages/vendorPackages:
 *   post:
 *     summary: Create a new vendor package
 *     description: Create a new vendor package
 *     tags: [VendorPackages]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               vendorId:
 *                 type: integer
 *               details:
 *                 type: string
 *               price:
 *                 type: integer
 *               bookingPrice:
 *                 type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       200:
 *         description: Successful operation
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *       '404':
 *         description: Vendor Packages not found.
 *       '500':
 *         description: Internal server error.
 */

router.post('/vendorPackages', Verify,async function (req, res) {
    try {
        await vendorPackagesService.createVendorPackage(req, res)
    }
    catch (e) {
        console.log(e);
    }
});

/**
* @swagger
*  /api/packages/vendorPackages:
*   get:
*     summary: Get all vendor packages
*     description: Get all vendor packages
*     tags: [VendorPackages]
*     security:
*       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
*     responses:
*       200:
*         description: Successful operation
*         content:
*           application/json:
*             schema:
*               type: array
*               items:
*                 $ref: '#/components/schemas/VendorPackage'
*       '404':
*         description: Vendor Packages not found.
*       '500':
*         description: Internal server error.
*/
router.get('/vendorPackages',Verify, async function (req, res) {
    try {
        await vendorPackagesService.getAllVendorPackages(req, res)
    }
    catch (e) {
        console.log(e);
    }
});

/**
 * @swagger
 * /api/packages/byVendorId:
 *   get:
 *     summary: Get vendor packages by vendor ID
 *     description: Get vendor packages by vendor ID
 *     tags: [VendorPackages]
 *     parameters:
 *       - in: query
 *         name: vendorId
 *         required: true
 *         schema:
 *           type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       200:
 *         description: Successful operation
 *       '404':
 *         description: Vendor Packages not found.
 *       '500':
 *         description: Internal server error.
 */
router.get('/byVendorId',Verify, async function (req, res) {
    try {
        await vendorPackagesService.getVendorPackagesByVendorId(req, res)
    }
    catch (e) {
        console.log(e);
    }
});

// Update
/**
 * @swagger
 * /api/packages/updatePackages:
 *   put:
 *     summary: Update vendor package
 *     description: Update vendor package
 *     tags: [VendorPackages]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               vendorId:
 *                 type: integer
 *               details:
 *                 type: string
 *               price:
 *                 type: integer
 *               bookingPrice:
 *                 type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       200:
 *         description: Successful operation
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/VendorPackage'
 *       '404':
 *         description: Vendor Packages not found.
 *       '500':
 *         description: Internal server error.
 */

router.put('/updatePackages',Verify, async function (req, res) {
    try {
        await vendorPackagesService.updateVendorPackage(req, res)
    }
    catch (e) {
        console.log(e);
    }
});

/**
 * @swagger
 * /api/packages/delete:
 *    delete:
 *     summary: Delete a vendor package
 *     description: Delete a vendor package by vendor ID and package ID
 *     tags: [VendorPackages]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *               packageId:
 *                 type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       200:
 *         description: Successful operation
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 message:
 *                   type: string
 *       400:
 *         description: Bad request - Missing vendorId or packageId
 *       404:
 *         description: Vendor package not found
 *       500:
 *         description: Internal server error
 */
router.delete('/delete',Verify, async function (req, res) {
    try {
        await vendorPackagesService.deleteVendorPackage(req, res)
    }
    catch (e) {
        console.log(e);
    }
});


module.exports = router;