const bcrypt = require('bcrypt');
const multer = require('multer');
const path = require('path');

const SALT_ROUNDS = 10;

getHashedPassword = (pwd) => {
    return bcrypt.hash(pwd, SALT_ROUNDS)
        .then(function (hash) {
            return hash;
        })
        .catch(function (e) {
            console.log(e);
            throw e;
        });
}

compareHash = (userPwd, accountPwd) => {
    return bcrypt.compare(userPwd, accountPwd)
    .then(function (res){
        return res;
    })
    .catch(function(e){
        throw e;
    })
}

parseJwtPayload = (token)=> {
    let base64Url = token.split('.')[1];
    let base64 = base64Url.replace('-', '+').replace('_', '/');
    return JSON.parse(Buffer.from(base64, 'base64'));
}

checkTokenExpiry = (token) => {
    let isExpiredToken = false;
    let dateNow = new Date();
    let decodedToken = parseJwtPayload(token)
    if (decodedToken.exp < dateNow.getTime() / 1000) {
        isExpiredToken = true;
    }
    console.log('TOKEN EXPIRED ? ', isExpiredToken)
    return isExpiredToken;
}

const formatDate = (date) => {
    const [dateStr] = new Date(date).toISOString().split('T')
    return dateStr
}

// 8. Upload Image Controller
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, 'Images')
    },
    filename: (req, file, cb) => {
        cb(null, Date.now() + path.extname(file.originalname))
    }
})

const upload = multer({
    storage: storage,
    limits: { fileSize: '10000000' },
    fileFilter: (req, file, cb) => {
        const fileTypes = /jpeg|jpg|png|gif/
        const mimeType = fileTypes.test(file.mimetype)
        const extname = fileTypes.test(path.extname(file.originalname))

        if (mimeType && extname) {
            return cb(null, true)
        }
        cb('Give proper files formate to upload')
    }
}).single('image')

const genResponse =  (res, code, msg, result = '') => {
    return res.status(code).json({
        message: msg,
        code: code,
        result
    });
}

const genPagingResponse =  (res, code, msg, result = '' , currentPage  , totalCount , pageSize) => {
    return res.status(code).json({
        message: msg,
        code: code,
        currentPage,
        totalPages :Math.ceil(totalCount / pageSize) ,
        totalRecords: totalCount,
        result
    });
}


const isEmpty = (obj) => {
    return obj == undefined || obj == null || obj == '';
}

const getLimitForPagination = (page, pageSize) => {
    page = page < 1 ? 1 : page;
    const startIndex = (page - 1) * pageSize;
    const endIndex = parseInt(startIndex) + parseInt(pageSize);
    return { start: parseInt(startIndex), end: parseInt(endIndex) }
}

function ensureArray(input) {
    if (Array.isArray(input)) {
      return input; // It's already an array, so return it as is
    } else {
      return [input]; // Wrap the object inside an array and return
    }
  }

// Utility function to generate all dates for a given month
const generateDatesForMonth = (year, month) => {
    const dates = [];
    const daysInMonth = new Date(year, month, 0).getDate(); // Get the number of days in the month
    for (let day = 1; day <= daysInMonth; day++) {
        const date = new Date(year, month-1, day+1); // Construct the Date object without adjusting the month
        dates.push(date.toISOString().split('T')[0]); // Convert to YYYY-MM-DD format
    }
    return dates;
};


module.exports = {
    getHashedPassword , 
    compareHash,
    parseJwtPayload,
    checkTokenExpiry,
    generateDatesForMonth,
    formatDate,
    upload,
    genResponse,
    isEmpty,
    getLimitForPagination,
    genPagingResponse, 
    ensureArray
};